{ Copyright (C) by Tobias Schwarz, 2003-2007
  Distributed under the GNU General Public License }

unit Webalize;

interface

uses StrUtils, Classes, SysUtils, 
{$IFDEF WIN32}
Forms, Dialogs, Controls, Windows, Registry, ConsoleApp, StdCtrls;
{$ELSE}
QControls, QForms, QDialogs, QStdCtrls, Libc, Inifiles;
{$ENDIF}

type
  TWebalize=class(TThread)
  private
    IncreasePos: Boolean;
    DoingWith, DoingWhat, Logfile: String;
    LogfileIndex: Integer;
    WebalizerOutput: TStringList;
    function PrepareLogfile(Logfile: String): String;
    function PreprocessLogfile(Filename: String): Boolean;
    function RunWebalizer(Logfile: String): Integer;
  protected
    procedure Execute; override;
    procedure UpdateForms;
    procedure ShowOutput;
  public
    Logfiles: TListBox;
    EnableFilter, MixDomains, RemoveDomain, SaveFilteredLog, CompressLog, RemoveHost: Boolean;
    WebalizerDir, Domain, SaveCompressedLogIn, TempDir, Hostname: String;
    Provider: Integer;
    constructor Create;
end;

function GetWebalizerDir: String;
function GetWebalizerVersion: Integer;
procedure WriteWebalizerDir(Dir: String);

implementation

uses Main, Resourcestrings, Split_Logfile, YclZlib;

constructor TWebalize.Create;
begin
inherited Create(true);
FreeOnTerminate:=true;
WebalizerOutput:=TStringList.Create;
end;

procedure TWebalize.UpdateForms;
begin
MainForm.ProgressForm.Doing_With_Label.Caption:=DoingWith;
MainForm.ProgressForm.Doing_What_Label.Caption:=ExtractFileName(DoingWhat);
If IncreasePos then MainForm.ProgressForm.ProgressBar.Position:=MainForm.ProgressForm.ProgressBar.Position+1;
If LogfileIndex>-1 then MainForm.LogList.Items[LogfileIndex]:=Logfile;
end;

procedure TWebalize.ShowOutput;
begin
MainForm.Output.Lines.Add(WebalizerOutput.Text);
end;

procedure TWebalize.Execute;
var Processfile: String;
    Ctr: Integer;
begin
For Ctr:=0 to Logfiles.Items.Count-1 do if Logfiles.Selected[Ctr] then begin
   Logfile:=Logfiles.Items[Ctr]; Delete(Logfile,1,1);      { Der Dateiname wird extrahiert }
   If FileExists(Logfile) then begin                      { Es wird berprft ob das Logfile berhaupt existiert }
      DoingWith:=ExtractFileName(Logfile);
      Processfile:=PrepareLogfile(Logfile);
      If Terminated then Exit;
      If (Processfile<>'') and (RunWebalizer(Processfile)=0) then Insert('1',Logfile,0) else Insert('2',Logfile,0);
      { Eine '1' vor dem Logfile wenn es analysiert wurde, eine '2' wenn ein Fehler aufgetreten ist }
      IncreasePos:=true; LogfileIndex:=Ctr;
      Synchronize(UpdateForms);
   end else MessageDlg(logfile+Logfile+could_not_find,mtInformation,[mbOK],0); { Fehlermeldung falls das Logfile nicht existiert }
   If Terminated then Exit;
end;
WebalizerOutput.Free;
end;

function TWebalize.PrepareLogfile(Logfile: String): String;
var SaveFileName: String;
    SplittedFileExists: Boolean;
begin
If EnableFilter or RemoveHost then begin   { Das Logfile wird ggfs gesplittet }
   SplittedFileExists:=false;

   { Wenn ein gespeichertes gesplittetes Logfile existiert, wird dieses fr die Analyse verwendet }
   If SaveFilteredLog then begin
      If ExtractFileExt(ExtractFilename(Logfile))='.gz' then SaveFileName:=ChangeFileExt(ExtractFilename(Logfile),'')
      else SaveFileName:=ExtractFilename(Logfile);
      If (CompressLog=false) and FileExists(SaveCompressedLogIn+SaveFileName) then begin
         Result:=SaveCompressedLogIn+SaveFileName;
         SplittedFileExists:=true;
      end else if CompressLog and FileExists(SaveCompressedLogIn+SaveFileName+'.gz') then begin
         Result:=SaveCompressedLogIn+SaveFileName+'.gz';
         SplittedFileExists:=true;
      end;
   end;

   If SplittedFileExists=false then begin
      If PreprocessLogfile(Logfile) then
         Result:=TempDir+'splitted_log.tmp' { Das gesplittete Logfile wird als splitted_log.tmp gespeichert }
         else Result:='';
   end;
end else Result:=Logfile;
DoingWhat:=analysing_logfile; IncreasePos:=true; LogfileIndex:=-1;
Synchronize(UpdateForms);
{ Result ist somit entweder das ursprngliche Logfile, ein bei einer vorherigen Analyse gesplittetes
  Logfile, die Datei splitted_log.tmp oder ein leerer String falls das Splitten fehlschlug }
end;

function TWebalize.PreprocessLogfile(Filename: String): Boolean;
var SaveFilename: String;
    CopyFrom, CopyTo: TFileStream;
    PreprocessLog: TPreprocessLog;
begin
Result:=true;
WebalizerOutput.Clear;
DoingWhat:=splitting_logfile; IncreasePos:=false; LogfileIndex:=-1;
Synchronize(UpdateForms);
PreprocessLog:=TPreprocessLog.Create;
WebalizerOutput.Add('Preprocessing logfile...');
If EnableFilter then begin
   PreprocessLog.SplitLog:=true;
   WebalizerOutput.Add('Filter domain: '+Domain);
   Case Provider of
      0: PreprocessLog.Provider:='1und1';
      3: PreprocessLog.Provider:='hosteurope';
   end;
   WebalizerOutput.Add('Provider: '+PreprocessLog.Provider);
   PreprocessLog.Mix:=MixDomains;
   PreprocessLog.Remove:=RemoveDomain;
end;
PreprocessLog.RemoveHost:=RemoveHost;
PreprocessLog.Hostname:=Hostname;
PreprocessLog.PreprocessLog(Filename,TempDir+'splitted_log.tmp',Domain);
WebalizerOutput.Add(IntToStr(PreprocessLog.Entries)+' records found');
If EnableFilter then WebalizerOutput.Add(IntToStr(PreprocessLog.Found)+' entries matching');
If RemoveHost then WebalizerOutput.Add(IntToStr(PreprocessLog.HostnamesFound)+' hostnames removed');
WebalizerOutput.Add('Processing time: '+IntToStr(PreprocessLog.ProcessTime)+' ms');
PreprocessLog.Free;

If PreprocessLog.Found=0 then begin
   MessageDlg(no_matching_records_found,mtError,[mbOK],0);
   Result:=false;
   Exit;
end;

If SaveFilteredLog then begin
   If SaveCompressedLogIn='' then begin
      MessageDlg(no_dir_for_splitted_log,mtError,[mbOK],0);
      Exit;
   end;
   WebalizerOutput.Add('Saving splitted logfile...');
   If ExtractFileExt(Filename)='.gz' then SaveFileName:=ChangeFileExt(ExtractFilename(Filename),'') else SaveFileName:=ExtractFilename(Filename);
   If CompressLog then begin

      DoingWith:=''; DoingWhat:=compressing_logfile; IncreasePos:=false; LogfileIndex:=0;
      Synchronize(UpdateForms);
      Try
         RenameFile(TempDir+'splitted_log.tmp',TempDir+SaveFileName);
         GZipCompressFile(TempDir+SaveFileName,SaveCompressedLogIn+SaveFileName+'.gz');
         RenameFile(TempDir+SaveFileName,TempDir+'splitted_log.tmp');
      except
         MessageDlg(compress_err,mtError,[mbOK],0);
         Exit;
      end;
   end else begin  { Wenn das gesplittete Logfile unkomprimiert gespeichert werden soll }
      CopyFrom:=TFileStream.Create(TempDir+'splitted_log.tmp',fmOpenRead);
      CopyTo:=TFileStream.Create(SaveCompressedLogIn+SaveFileName,fmCreate);
      CopyTo.CopyFrom(CopyFrom,CopyFrom.size);
      CopyFrom.Free;
      CopyTo.Free;
   end;
end;
Synchronize(ShowOutput);
end;

function TWebalize.RunWebalizer(Logfile: String): Integer; { Webalizer ausfhren }
{$IFDEF WIN32}
var CAExitCode, i, TxtPos: Integer;
    OutText: String;
const Messages: Array [1..13] of String = ('Using ','DNS workers','Creating output','Hostname for reports','History file',
                                           'Reading history','Previous run','Reading previous','Saving current','Generating ',
                                           'DNS cache hit','Saving history','No valid records');
begin
WebalizerOutput.Clear;
CAExitCode:=ExecConsoleApp(WebalizerDir+'Webalizer.exe','-c "'+TempDir+'webalizer.conf" "'+Logfile+'"',WebalizerOutput,nil);
OutText:=WebalizerOutput.Text; { Die Ausgabe von Webalizer wird formatiert, da sonst alles in einer Zeile steht }
While Pos(#13#10,OutText)>0 do Delete(OutText,Pos(#13#10,OutText),2);
For i:=1 to 13 do begin
   TxtPos:=Pos(Messages[i],OutText);
   While TxtPos>0 do begin
      Insert(#13#10,OutText,TxtPos);
      TxtPos:=PosEx(Messages[i],OutText,TxtPos+3);
   end;
end;
If Pos('Saving history',OutText)>0 then Insert(#13#10,OutText,Pos('Saving history',OutText)+29);
WebalizerOutput.Text:=OutText;
WebalizerOutput.Add(Format('%s returned %d', [WebalizerDir+'Webalizer.exe', CAExitCode]));
Result:=CAExitCode;
{$ELSE}
var Output: PIOFile; { neuer Standart Output Stream }
    line: PChar;
    txt, str: String;
    rb: Integer;
const
    BufferSize: Integer = 1000;
begin
WebalizerOutput.Clear;
SetLength(txt,0);
str:=WebalizerDir+'webalizer -c '+GetEnvironmentVariable('HOME')+'/.webalizer.conf '+Logfile;
Output:=popen(PChar(str),'r');    { Hidden ecexution of Arg <pCMD> }
GetMem(Line,BufferSize);
{ Ausgabe von <Output> in das OutputMemo eintragen und dabei Zeilenumbrche [Chr(10)] vornehmen. }
If Assigned(Output) then
   While FEOF(Output)=0 do begin
      rb:=libc.fread(line,1,BufferSize,Output);
      SetLength(txt,length(txt)+rb);
      MemCpy(@txt[length(txt)-(rb-1)],line,rb);
      while pos(#10,txt)>0 do begin
         str:=copy(txt,1,pos(#10,txt)-1);
         WebalizerOutput.Add(str);
         txt:=copy(txt,pos(#10,txt)+1,length(txt));
      end;
   end;
libc.pclose(Output);   { Output-Stream wieder schliessen }
wait(nil);
FreeMem(Line,BufferSize);
Result:=0;
{$ENDIF}
Synchronize(ShowOutput);
If Result<>0 then MessageDlg(webalizer_exec_err,mtError,[mbOK],0);
end;

function GetWebalizerDir: String;
{$IFDEF WIN32}
var WebalizerDirExists: Boolean;
    Registry: TRegistry;
begin
Registry:=TRegistry.Create;
Registry.RootKey:=HKEY_CURRENT_USER;
WebalizerDirExists:=false;
If (Registry.KeyExists('Software\Sir_Tobe&Co\WebalizerGUI\ProgramProperty')) and (Registry.OpenKeyReadOnly('Software\Sir_Tobe&Co\WebalizerGUI\ProgramProperty')) then begin
   If Registry.ValueExists('WebalizerExe') then Result:=Registry.ReadString('WebalizerExe'); { Das Verzeichnis in dem Webalizer.exe liegt wird geladen }
   Registry.CloseKey;
   If FileExists(Result+'webalizer.exe') then WebalizerDirExists:=true;
end;
If WebalizerDirExists=false then begin
   Registry.RootKey:=HKEY_LOCAL_MACHINE;
   If (Registry.KeyExists('Software\Sir_Tobe&Co\WebalizerGUI\ProgramProperty')) and (Registry.OpenKeyReadOnly('Software\Sir_Tobe&Co\WebalizerGUI\ProgramProperty')) then begin
      If Registry.ValueExists('WebalizerExe') then Result:=Registry.ReadString('WebalizerExe'); { Das Verzeichnis in dem Webalizer.exe liegt wird geladen }
      Registry.CloseKey;
      If FileExists(Result+'webalizer.exe') then WebalizerDirExists:=true;
   end;
end;
Registry.Free;
If WebalizerDirExists=false then begin    { Falls noch kein Eintrag fr das Verzeichnis von webalizer.exe existiert oder die Datei dort nicht gefunden werden kann }
    If FileExists(ExtractFilePath(Application.ExeName)+'webalizer.exe') then   { Falls Webalizer im gleichen Verzeichnis wie die GUI liegt ist alles OK }
       Result:=ExtractFilePath(Application.ExeName)
    else begin                        { Anderenfalls mu der Anwender das Verzeichnis in dem Webalizer liegt angeben }
      Case MessageDlg(select_webalizer_exe,mtConfirmation,[mbOK,mbCancel],0) of
         mrCancel: Application.Terminate;
         mrOK: begin
             MainForm.OpenDialog.InitialDir:=HomeDir;
             MainForm.OpenDialog.FileName:='webalizer.exe';
             If MainForm.OpenDialog.Execute then begin   { Das Verzeichnis in dem webalizer.exe liegt wird bestimmt }
                If FileExists(ExtractFilePath(MainForm.OpenDialog.Filename)+'webalizer.exe') then Result:=ExtractFilePath(MainForm.OpenDialog.Filename) else Application.Terminate;
             end else Application.Terminate;
         end;
      end;
    end;
    WriteWebalizerDir(Result);
end;
{$ELSE}
var WebalizerExists: Boolean;
    Inifile: TInifile;
begin
IniFile:=TIniFile.Create(GetEnvironmentVariable('HOME')+'/.xwebalizer');
WebalizerExists:=false;
Result:=IniFile.ReadString('ProgramProperty','WebalizerExe',ExtractFilePath(Application.ExeName)); { Das Verzeichnis in dem Webalizer.exe liegt wird geladen }
IniFile.Free;
If FileExists(Result) then WebalizerExists:=true;
If WebalizerExists=false then begin         { Falls noch kein Eintrag fr das Verzeichnis von webalizer existiert oder die Datei dort nicht gefunden werden kann }
   If FileExists('/usr/bin/webalizer') then Result:='/usr/bin/'
   else if FileExists('/usr/local/bin/webalizer') then Result:='/usr/local/bin/'
   else if FileExists('/usr/X11/bin/webalizer') then Result:='/usr/X11/bin/'
   else if FileExists('/bin/webalizer') then Result:='/bin/'
   else if FileExists('/usr/local/xwebalizer/webalizer') then Result:='/usr/local/xwebalizer/'
   else if FileExists('/opt/xwebalizer/webalizer') then Result:='/opt/xwebalizer/'
   else begin                          { Anderenfalls mu der Anwender das Verzeichnis in dem Webalizer liegt angeben }
      Case MessageDlg(select_webalizer_exe,mtConfirmation,[mbOK,mbCancel],0) of
         mrCancel: Application.Terminate;
         mrOK: begin
             If DirectoryExists('/usr/bin') then MainForm.OpenDialog.InitialDir:='/usr/bin' else MainForm.OpenDialog.InitialDir:='/';
             If MainForm.OpenDialog.Execute then begin   { Das Verzeichnis in dem Webalizer liegt wird bestimmt }
                If FileExists(MainForm.OpenDialog.Filename) then Result:=ExtractFileDir(MainForm.OpenDialog.Filename) else Application.Terminate;
             end else Application.Terminate;
         end;
      end;
    end;
end;
{$ENDIF}
end;

procedure WriteWebalizerDir(Dir: String);
{$IFDEF WIN32}
var Registry: TRegistry;
begin
Registry:=TRegistry.Create;
Registry.RootKey:=HKEY_LOCAL_MACHINE;
Registry.Access:=KEY_ALL_ACCESS;
If Registry.OpenKey('Software\Sir_Tobe&Co\WebalizerGUI\ProgramProperty',true) then begin
   Try Registry.WriteString('WebalizerExe',Dir);
   Finally Registry.CloseKey; end;
end;
Registry.RootKey:=HKEY_CURRENT_USER;
If Registry.OpenKey('Software\Sir_Tobe&Co\WebalizerGUI\ProgramProperty',true) then begin
   Try Registry.WriteString('WebalizerExe',Dir);
   Finally Registry.CloseKey; end;
end;
Registry.Free;
{$ELSE}
var IniFile: TIniFile;
begin
IniFile:=TIniFile.Create(GetEnvironmentVariable('HOME')+'/.xwebalizer');
IniFile.WriteString('ProgramProperty','WebalizerExe',Dir);
IniFile.UpdateFile;
IniFile.Free;
{$ENDIF}
end;

function GetWebalizerVersion: Integer;
var OutText: String;
    ExitCode: Integer;

{$IFDEF WIN32}
    WebalizerOutput: TStringList;
begin
Result:=9999999;
WebalizerOutput:=TStringList.Create;
ExitCode:=ExecConsoleApp(MainForm.WebalizerDir+'Webalizer.exe','-v',WebalizerOutput,nil);
If ExitCode<>0 then OutText:=WebalizerOutput.Text;
WebalizerOutput.Free;

{$ELSE}
    Output: PIOFile; { neuer Standart Output Stream }
    line: PChar;
    str: String;
    rb: Integer;
const
    BufferSize: Integer = 1000;
begin
Result:=9999999;
ExitCode:=0;
SetLength(OutText,0);
str:=MainForm.WebalizerDir+'webalizer -v';
Output:=popen(PChar(str),'r');
GetMem(Line,BufferSize);
If Assigned(Output) then begin
   While FEOF(Output)=0 do begin
      rb:=libc.fread(line,1,BufferSize,Output);
      If rb=0 then rb=100; { Bugfix fr ein seltsames Compilat von Webalizer V.2.01-10 }
      SetLength(OutText,length(OutText)+rb);
      MemCpy(@OutText[length(OutText)-(rb-1)],line,rb);
   end;
   ExitCode:=1;
end;
libc.pclose(Output);
wait(nil);
FreeMem(Line,BufferSize);
{$ENDIF}

If ExitCode<>0 then begin
   {$IFDEF WIN32}
   While Pos(#13#10,OutText)>0 do Delete(OutText,Pos(#13#10,OutText),2);
   {$ELSE}
   While Pos(#10,OutText)>0 do Delete(OutText,Pos(#10,OutText),1);
   {$ENDIF}
   If Pos('Copyright 1997',OutText)>0 then Insert(' ',OutText,Pos('Copyright 1997',OutText));
   If Pos('Copyright (c)',OutText)>0 then Insert(' ',OutText,Pos('Copyright (c)',OutText));
   If Pos('This',OutText)>0 then Insert(' ',OutText,Pos('This',OutText));
   MainForm.WebalizerExecutable.Text:=OutText;
   If Pos('Stone Steps',OutText)>0 then begin
      { The version information of the Stone Steps Webalizer contains 'based on Webalizer V. 2.01-10', therefore we have to treat these versions separately }
      If Pos('v2.2.',OutText)>0 then Result:=2020000; { V2.2.0 oder hher von Stone Steps }
      If Pos('v2.3.',OutText)>0 then Result:=2030000; { V2.3.0 oder hher von Stone Steps }
      If Pos('v2.4.',OutText)>0 then Result:=2040000; { V2.4.0 oder hher von Stone Steps }
      If Pos('v2.5.',OutText)>0 then Result:=2050000; { V2.5.0 oder hher von Stone Steps }
      If Pos('v2.6.',OutText)>0 then Result:=2060000; { V2.6.0 oder hher von Stone Steps }
      If Pos('v2.7.',OutText)>0 then Result:=2070000; { V2.7.0 oder hher von Stone Steps }
      If Pos('v2.8.',OutText)>0 then Result:=2080000; { V2.8.0 oder hher von Stone Steps }
      If Pos('v3.',OutText)>0 then Result:=3000000; { V3.0.0 oder hher von Stone Steps }
   end else begin
      If Pos('2.01-10',OutText)>0 then Result:=2011000; { V2.01-10 von Brad Barrett }
      If (Pos('2.01-10-RB02',OutText)>0) or (Pos('2.01-10 RB02',OutText)>0) then Result:=2011002; { V2.01-10-RB02 von Medasys-Lille }
      If (Pos('2.01-10-RB1',OutText)>0) or (Pos('2.01-10 RB1',OutText)>0) then Result:=2011010; { V2.01-10-RB10 oder hher von Frey Development }
   end;
end;
MainForm.WebalizerVersion:=Result;
MainForm.SetWebalizerSpecifics(Result);
end;

end.
