unit WildMatch;

//***********************************************************************
//*    TWildMatch component by Elliott Shevin  shevine@aol.com
//*
//*    This component is freeware. I'd appreciate a credit if you use it.
//*
//*    TWildMatch encapsulates a routine to compare two strings, one of
//*    which can contain "wild card" characters. Just as in DOS, Windows,
//*    and Unix, two wild cards are available: one which matches any 
//*    single character, and one that matches zero or more characters.
//*
//*    You may find this component most useful in gathering directory
//*    and file names that fall in a range of possible values.
//*    If you want to replicate file searches as Windows does them,
//*    remember that if you don't specify a search string that
//*    limits the acceptable filename extensions, Windows assumes
//*    '.*'--i.e., that all extensions are acceptable. Using this
//*    routine, you would have to adjust your mask string to compensate.
//*
//*    TWildMatch is suitable for all versions of Delphi. Install it as
//*    you would any other source code component. TWildMatch is nonvisible.
//*
//*    Properties:
//*      WildString   not quite what it sounds like--it's a single character
//*                   which is the wild card that matches zero or more
//*                   characters as necessary. Default is '*'.
//*      WildChar     the wild card that matches any single character.
//*                   Default is '?'.
//*      MatchCase    boolean; TRUE if comparisons are to be case-sensitive.
//*                   Default is FALSE.
//*
//*    Methods:
//*      Matching(SearchString, Mask : string) : boolean;
//*          Returns TRUE if the two strings match, and FALSE if not. 
//*          Mask uses the wild cards defined in WildString and WildChar, 
//*          and the value of MatchCase controls case sensitivity.
//*
//*    Events: none.
//***************************************************************************

interface

//***********************************************************************
//*  I would be very surprised if the Graphics, Controls, Forms, and
//*  Dialogs units are necessary for this component.
//***********************************************************************
uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

function Matching(SearchString, Mask : string) : boolean;

implementation

function Matching(SearchString, Mask : string) : boolean;
var
   s, m   : string;
   ss     : string[1];
   c      : char;
begin
   s  := Uppercase(SearchString);
   m  := Uppercase(Mask);
   // Compare the two strings one character at a time until one or the
   // other is exhausted. For each character that matches, we truncate
   // both strings at the left.
   while (length(s) > 0) and (length(m) > 0) do begin

      // Take the first character from the mask.
      ss := copy(m,1,1);
      c := ss[1];

         // If the next mask character is wildchar, count the two characters
         // as matching; lop them off both strings.
      if c = '?' then begin
                  delete(s,1,1);
                  delete(m,1,1);
                end

         // If the next character is a WildString, lop it off the
         // mask string, since it matches whatever follows.
         // Then keep calling this routine recursively
         // to see if what's left of the search string matches what
         // remains of the mask. If it doesn't, truncate the search
         // string at the left (the WildString character matches
         // those bytes, too) and repeat until either there's a match
         // or the search string is exhausted, which means the
         // WildString character has eaten the entire remaining
         // search string.
      else
         if c = '*' then begin
                   delete(m,1,1);
                   while (not Matching(s,m))
                               and (length(s) > 0) do
                       delete(s,1,1);
         end

         // Any other character must be an exact match. If not, 
         // clear the search string to stop the loop, as a match
         // failure has been detected. This will be sealed below
         // because the search string is null but the mask string
         // is not.
         else if copy(s,1,1) = copy(m,1,1)
                 then begin
                        delete(s,1,1);
                        delete(m,1,1);
                 end
                 else s := '';
   end;

   // If the loop is ended, the strings have matched if they
   // are now both reduced to null or if the match string
   // is reduced to a single WildString character.
   result := ((length(s) = 0) and (length(m) = 0))
                or (m = '*');
end;

end.
